/*
  assets/js/editor.js
  Reverted to "Always New Version on Save"
  Version: 2025-02-12

  - Pressing "Save" on an existing doc => calls save_document.php with "save_as_new=1".
  - We do not overwrite the old version => no data loss
  - Hides "Copy as New" icon
  - Compare Mode auto expands/collapses doc listings
  - Unsaved => immediate Save
  - Press "Enter" to finalize rename
*/

$(document).ready(function(){
    var newDocCounter = 0;
    var compareModeActive = false;
    var selectedCompareVersions = [];

    // UTILS
    function escapeHtml(text){
        return $('<div>').text(text).html();
    }
    function createEditorWrapper(content){
        var container=$('<div class="editor-container"></div>');
        var $lineNumbers=$('<div class="line-numbers"></div>');
        var $textarea=$('<textarea class="editor-textarea"></textarea>');
        $textarea.val(content);
        $textarea.data('original',content);
        container.append($lineNumbers).append($textarea);
        updateLineNumbers($textarea);
        $textarea.on('scroll',function(){
            $lineNumbers.scrollTop($textarea.scrollTop());
        });
        return container;
    }
    function updateLineNumbers($textarea){
        var lines = $textarea.val().split(/\r\n|\n|\r/).length;
        var $ln = $textarea.siblings('.line-numbers');
        var lineText='';
        for(var i=1;i<=lines;i++){
            lineText += i+'<br>';
        }
        $ln.html(lineText);
    }
    function updateStatusBar($pane){
        var $textarea = $pane.find('.editor-textarea');
        var content = $textarea.val();
        var lineCount = content.split(/\r\n|\n|\r/).length;
        var charCount = content.length;
        $pane.find('.line-count').text(lineCount);
        $pane.find('.char-count').text(charCount);
    }
    function updateSaveButtonState(){
        var $active = $('#tabContent .tab-pane.active');
        if($active.length){
            var $textarea = $active.find('.editor-textarea');
            var isUnchanged = ($textarea.val()=== $textarea.data('original'));
            $('#saveDocButton').prop('disabled', isUnchanged);
        }
    }

    // ADD TAB
    function addTab(tabId, title, content, dataObj){
        var $li = $('<li class="nav-item" role="presentation"></li>');
        var $btn= $('<button class="nav-link doc-tab" type="button" role="tab"></button>');
        $btn.attr('id', tabId+'-tab');
        $btn.attr('data-bs-target','#'+tabId);
        $btn.attr('data-bs-toggle','tab');
        if(dataObj){ $btn.data(dataObj); }

        var headerHtml = '<span class="doc-title">'+escapeHtml(title)
            +'</span> (v'+(dataObj.version||1)+') <span class="close-tab" style="cursor:pointer;">&times;</span>';
        $btn.html(headerHtml);
        $li.append($btn);
        $('#documentTabs').append($li);

        var $pane = $('<div class="tab-pane fade" role="tabpanel"></div>');
        $pane.attr('id', tabId);
        if(dataObj){ $pane.data(dataObj); }

        var $wrapper=createEditorWrapper(content);
        var $statusBar = $('<div class="editor-status">Lines: <span class="line-count">0</span> | Characters: <span class="char-count">0</span></div>');
        $statusBar.css({
            padding:'5px',
            'font-size':'0.9em',
            background:'#f8f9fa',
            'border-top':'1px solid #ddd',
            'text-align':'right'
        });
        $pane.append($wrapper).append($statusBar);
        $('#tabContent').append($pane);
        updateStatusBar($pane);
    }

    // OPEN EXISTING DOC
    function openExistingDocumentTab(docId, version, title){
        var tabId='tab-'+docId+'-'+version;
        if(!$('#'+tabId).length){
            addTab(tabId,title,'Loading...', {docId:docId, version:version, isNew:false});
            $.ajax({
                url:'api/load_document.php',
                method:'GET',
                data:{document_id:docId, version:version},
                dataType:'json',
                success:function(resp){
                    var $pane = $('#'+tabId);
                    var $textarea=$pane.find('.editor-textarea');
                    if(resp.success){
                        $textarea.val(resp.content);
                        if(resp.status && resp.status==='unsaved'){
                            $textarea.data('original','__unsaved_marker__');
                        } else {
                            $textarea.data('original',resp.content);
                        }
                    } else {
                        $textarea.val('Error loading document.');
                    }
                    updateLineNumbers($textarea);
                    updateStatusBar($pane);
                    updateSaveButtonState();
                },
                error:function(){
                    $('#'+tabId).find('.editor-textarea').val('Error loading document.');
                }
            });
        }
        var triggerEl=new bootstrap.Tab(document.getElementById(tabId+'-tab'));
        triggerEl.show();
        $('#placeholderMessage').hide();
    }

    // OPEN NEW
    function openNewDocumentTab(title, content){
        newDocCounter++;
        var docId='new-'+newDocCounter;
        var tabId='tab-'+docId;
        addTab(tabId,title,content,{docId:0, version:1, isNew:true});
        var triggerEl=new bootstrap.Tab(document.getElementById(tabId+'-tab'));
        triggerEl.show();
        $('#placeholderMessage').hide();
    }

    // autoSaveAllOpenDocsExcept
    function autoSaveAllOpenDocsExcept($excludePane){
        return new Promise(function(resolve,reject){
            var tasks=[];
            $('#tabContent .tab-pane').each(function(){
                var $pane=$(this);
                if($excludePane && $pane[0]===$excludePane[0])return;
                var data=$pane.data();
                var $textarea=$pane.find('.editor-textarea');
                if($textarea.val()!==$textarea.data('original')){
                    var content=$textarea.val();
                    if(data.isNew && data.docId===0){
                        // create doc => version_label:'(autosave)'
                        tasks.push(function(){
                            return new Promise(function(res,rej){
                                $.ajax({
                                    url:'api/create_document.php',
                                    method:'POST',
                                    data:{
                                        title:($pane.find('.doc-title').text()||'Untitled'),
                                        content:content,
                                        version_label:'(autosave)',
                                        csrf_token:$('meta[name="csrf-token"]').attr('content')
                                    },
                                    dataType:'json',
                                    success:function(r){
                                        if(r.success){
                                            $pane.data('docId',r.document_id);
                                            $pane.data('version',r.version);
                                            $pane.data('isNew',false);
                                            $textarea.data('original',content);
                                            res();
                                        } else {
                                            rej('Autosave create doc error: '+r.message);
                                        }
                                    },
                                    error:function(){
                                        rej('Autosave create doc request failed.');
                                    }
                                });
                            });
                        });
                    } else {
                        // existing => always new => save_as_new=1
                        tasks.push(function(){
                            return new Promise(function(res,rej){
                                $.ajax({
                                    url:'api/save_document.php',
                                    method:'POST',
                                    data:{
                                        document_id:data.docId,
                                        version:data.version,
                                        content:content,
                                        save_as_new:1,
                                        version_label:'(autosave)',
                                        csrf_token:$('meta[name="csrf-token"]').attr('content')
                                    },
                                    dataType:'json',
                                    success:function(r){
                                        if(r.success){
                                            $pane.data('version',r.new_version);
                                            $textarea.data('original',content);
                                            res();
                                        } else {
                                            rej('Autosave save doc error: '+r.message);
                                        }
                                    },
                                    error:function(){
                                        rej('Autosave save doc request failed.');
                                    }
                                });
                            });
                        });
                    }
                }
            });
            (async function(){
                for(let fn of tasks){
                    await fn();
                }
                resolve();
            })().catch(function(err){
                reject(err);
            });
        });
    }

    // Double-click rename
    $('#documentTabs').on('dblclick','.doc-title',function(e){
        e.stopPropagation();
        var $span=$(this);
        var oldTitle=$span.text();
        var $input=$('<input type="text" class="form-control form-control-sm inline-title-edit" style="width:150px;">').val(oldTitle);
        $span.replaceWith($input);
        $input.focus();
        $input.on('keydown',function(evt){
            if(evt.key==='Enter'){
                evt.preventDefault();
                $input.blur();
            }
        });
        $input.on('blur', async function(){
            var newTitle=$input.val().trim()||oldTitle;
            var $tabBtn=$input.closest('button.doc-tab');
            $tabBtn.find('.doc-title').text(newTitle);
            $input.replaceWith($('<span class="doc-title">').text(newTitle));
            var data=$tabBtn.data();
            if(data.isNew && data.docId===0){
                try {
                    var sel=$tabBtn.attr('data-bs-target');
                    var $pane=$(sel);
                    await autoSaveAllOpenDocsExcept($pane);

                    var $textarea=$pane.find('.editor-textarea');
                    var content=$textarea.val();
                    $.ajax({
                        url:'api/create_document.php',
                        method:'POST',
                        data:{
                            title:newTitle,
                            content:content,
                            csrf_token:$('meta[name="csrf-token"]').attr('content')
                        },
                        dataType:'json',
                        success:function(r){
                            if(r.success){
                                location.reload();
                            } else {
                                showErrorModal('Error creating doc: '+r.message);
                            }
                        },
                        error:function(){
                            showErrorModal('Error create doc request.');
                        }
                    });
                } catch(err){
                    showErrorModal(err);
                }
            } else {
                // rename existing
                if(data.docId && data.docId!==0){
                    $.ajax({
                        url:'api/update_document_title.php',
                        method:'POST',
                        data:{
                            document_id:data.docId,
                            title:newTitle,
                            csrf_token:$('meta[name="csrf-token"]').attr('content')
                        },
                        dataType:'json',
                        error:function(){
                            showErrorModal('Error renaming doc in DB.');
                        }
                    });
                }
            }
        });
    });

    // close tab
    $('#documentTabs').on('click','.close-tab',function(e){
        e.stopPropagation();
        var $tabBtn=$(this).closest('button.doc-tab');
        var tgt=$tabBtn.attr('data-bs-target');
        $tabBtn.closest('li').remove();
        $(tgt).remove();
        if($('#documentTabs .doc-tab').length===0){
            $('#placeholderMessage').show();
        } else {
            var $firstTab=$('#documentTabs .doc-tab').first();
            new bootstrap.Tab(document.getElementById($firstTab.attr('id'))).show();
        }
        updateSaveButtonState();
    });

    // NEW & SAVE
    $('#newDocButton').on('click',function(){
        openNewDocumentTab('Untitled','Hello World');
    });

    $('#saveDocButton').on('click',function(){
        var $active=$('#tabContent .tab-pane.active');
        if(!$active.length)return;
        var data=$active.data();
        var $textarea=$active.find('.editor-textarea');
        var content=$textarea.val();

        if(data.isNew && data.docId===0){
            // brand new => create doc
            $.ajax({
                url:'api/create_document.php',
                method:'POST',
                data:{
                    title:($active.find('.doc-title').text()||'Untitled'),
                    content:content,
                    csrf_token:$('meta[name="csrf-token"]').attr('content')
                },
                dataType:'json',
                success:function(r){
                    if(r.success){
                        $active.data('docId',r.document_id);
                        $active.data('version',r.version);
                        $active.data('isNew',false);
                        var tabId=$active.attr('id');
                        var $tabBtn=$('#'+tabId+'-tab');
                        $tabBtn.data('docId',r.document_id);
                        $tabBtn.data('version',r.version);
                        $tabBtn.data('isNew',false);
                        $tabBtn.html('<span class="doc-title">'+escapeHtml(r.title)
                          +'</span> (v'+r.version+') <span class="close-tab">&times;</span>');
                        $textarea.data('original',content);
                        $('#saveSuccessModal').modal('show');
                    } else {
                        showErrorModal('Error creating doc: '+r.message);
                    }
                },
                error:function(){
                    showErrorModal('Error creating doc request.');
                }
            });
        } else {
            // existing => ALWAYS NEW => save_as_new=1 => never lose data
            $.ajax({
                url:'api/save_document.php',
                method:'POST',
                data:{
                    document_id:data.docId,
                    version:data.version,
                    content:content,
                    save_as_new:1, // always new
                    csrf_token:$('meta[name="csrf-token"]').attr('content')
                },
                dataType:'json',
                success:function(r){
                    if(r.success){
                        $active.data('version',r.new_version);
                        var tabId=$active.attr('id');
                        var $tabBtn=$('#'+tabId+'-tab');
                        var curTitle=$tabBtn.find('.doc-title').text();
                        $tabBtn.html('<span class="doc-title">'+escapeHtml(curTitle)
                          +'</span> (v'+r.new_version+') <span class="close-tab">&times;</span>');
                        $textarea.data('original',content);
                        $('#saveSuccessModal').modal('show');
                    } else {
                        // If the backend references "Document version mismatch," we can handle it or just show error
                        showErrorModal('Error saving doc: '+r.message);
                    }
                },
                error:function(){
                    showErrorModal('Error saving doc request.');
                }
            });
        }
    });

    $('a[data-bs-toggle="tab"]').on('shown.bs.tab',function(){
        updateSaveButtonState();
    });
    $('#saveSuccessOkButton').on('click',function(){
        location.reload();
    });

    $(document).on('input','.editor-textarea',function(){
        var $this=$(this);
        updateLineNumbers($this);
        var $pane=$this.closest('.tab-pane');
        updateStatusBar($pane);
        updateSaveButtonState();
    });

    // expand/collapse docs
    $('.toggle-history').on('click',function(){
        var $icon=$(this);
        var docId=$icon.data('doc-id');
        var $container=$('#history-'+docId);
        if($container.is(':visible')){
            $container.slideUp();
            $icon.removeClass('bi-dash').addClass('bi-plus');
        } else {
            if(!$container.data('loaded')){
                $.ajax({
                    url:'api/get_history.php',
                    method:'GET',
                    data:{document_id:docId},
                    dataType:'json',
                    success:function(resp){
                        if(resp.success){
                            var html='';
                            resp.versions.forEach(function(v){
                                html+='<div class="version-item" data-version-id="'+v.id+'" data-version-label="'+v.version+'">';
                                html+='  <div class="d-flex justify-content-between align-items-center">';
                                html+='    <div class="version-check" style="display:'+(compareModeActive?'block':'none')+';">';
                                html+='      <input type="checkbox" class="compare-checkbox" value="'+v.version+'" data-version-id="'+v.id+'"/>';
                                html+='    </div>';
                                html+='    <span class="open-version" data-doc-id="'+docId+'" data-version="'+v.version+'" style="cursor:pointer;">';
                                html+= v.saved_at+' - '+v.version+(v.is_current?' (current)':'');
                                html+='    </span>';
                                html+='    <span>';
                                html+='      <i class="bi bi-eye view-version" data-doc-id="'+docId+'" data-version="'+v.version+'" style="cursor:pointer;"></i> ';
                                // Hide "Copy as New"
                                // html+=' <i class="bi bi-files copy-version"...>';
                                html+='      <i class="bi bi-x del-ver-btn" data-doc-id="'+docId+'" data-version-id="'+v.id+'" style="cursor:pointer;"></i>';
                                html+='    </span>';
                                html+='  </div>';
                                html+='</div>';
                            });
                            $container.html(html);
                            $container.data('loaded',true);
                        } else {
                            $container.html('<p>Error loading versions.</p>');
                        }
                    },
                    error:function(){
                        $container.html('<p>Error loading versions.</p>');
                    }
                });
            }
            $container.slideDown();
            $icon.removeClass('bi-plus').addClass('bi-dash');
        }
    });

    $(document).on('click','.open-version', function(){
        var docId=$(this).data('doc-id');
        var version=$(this).data('version');
        var title=$(this).closest('.history-item').data('title')||'Document';
        openExistingDocumentTab(docId,version,title);
    });

    // del ver => confirm
    $(document).on('click','.del-ver-btn',function(){
        var docId=$(this).data('doc-id');
        var verId=$(this).data('version-id');
        $('#confirmDeleteVersionModal').data('docId',docId);
        $('#confirmDeleteVersionModal').data('versionId',verId);
        $('#confirmDeleteVersionModal').modal('show');
    });
    $('#confirmDeleteVersionButton').on('click',function(){
        var docId=$('#confirmDeleteVersionModal').data('docId');
        var verId=$('#confirmDeleteVersionModal').data('versionId');
        $('#confirmDeleteVersionModal').modal('hide');
        $.ajax({
            url:'api/delete_version.php',
            method:'POST',
            data:{
                document_id:docId,
                version_id:verId,
                csrf_token:$('meta[name="csrf-token"]').attr('content')
            },
            dataType:'json',
            success:function(resp){
                if(resp.success){
                    location.reload();
                } else {
                    showErrorModal('Error deleting version: '+resp.message);
                }
            },
            error:function(){
                showErrorModal('Error deleting version request.');
            }
        });
    });

    // del doc => confirm
    $(document).on('click','.delete-document-history',function(e){
        e.stopPropagation();
        var docId=$(this).data('doc-id');
        $('#confirmDeleteDocumentModal').data('docId',docId);
        $('#confirmDeleteDocumentModal').modal('show');
    });
    $('#confirmDeleteDocumentButton').on('click',function(){
        var docId=$('#confirmDeleteDocumentModal').data('docId');
        $('#confirmDeleteDocumentModal').modal('hide');
        $.ajax({
            url:'api/delete_document.php',
            method:'POST',
            data:{
                document_id:docId,
                delete_main:true,
                csrf_token:$('meta[name="csrf-token"]').attr('content')
            },
            dataType:'json',
            success:function(resp){
                if(resp.success){
                    location.reload();
                } else {
                    showErrorModal('Error deleting doc: '+resp.message);
                }
            },
            error:function(){
                showErrorModal('Error deleting doc request.');
            }
        });
    });

    // compare mode
    $('#compareToggleBtn').on('click',function(){
        compareModeActive=!compareModeActive;
        selectedCompareVersions=[];
        $('.history-versions .version-check').toggle(compareModeActive);
        $('#compareSelectedBtn').hide();

        if(compareModeActive){
            // expand all
            $('.toggle-history').each(function(){
                var $icon=$(this);
                var did=$icon.data('doc-id');
                var $cont=$('#history-'+did);
                if(!$cont.is(':visible')){
                    $icon.trigger('click');
                }
            });
        } else {
            // collapse all
            $('.toggle-history').each(function(){
                var $icon=$(this);
                var did=$icon.data('doc-id');
                var $cont=$('#history-'+did);
                if($cont.is(':visible')){
                    $icon.trigger('click');
                }
            });
        }
    });
    $(document).on('change','.compare-checkbox', function(){
        if(!compareModeActive)return;
        var $this=$(this);
        var verId=$this.data('version-id');
        var versionLabel=$this.val();
        var docId=$this.closest('.history-versions').attr('id').split('-')[1];
        if($this.is(':checked')){
            if(selectedCompareVersions.length>=2){
                $this.prop('checked',false);
                return;
            }
            selectedCompareVersions.push({
                docId:docId,
                versionId:verId,
                versionLabel:versionLabel
            });
        } else {
            selectedCompareVersions=selectedCompareVersions.filter(function(o){
                return o.versionId!==verId;
            });
        }
        if(selectedCompareVersions.length===2){
            $('#compareSelectedBtn').show();
        } else {
            $('#compareSelectedBtn').hide();
        }
    });
    $('#compareSelectedBtn').on('click',function(){
        if(selectedCompareVersions.length!==2)return;
        var docA=selectedCompareVersions[0].docId;
        var vA=selectedCompareVersions[0].versionLabel;
        var docB=selectedCompareVersions[1].docId;
        var vB=selectedCompareVersions[1].versionLabel;
        $.ajax({
            url:'api/compare_versions.php',
            method:'POST',
            data:{
                doc_id1:docA,
                version1:vA,
                doc_id2:docB,
                version2:vB,
                csrf_token:$('meta[name="csrf-token"]').attr('content')
            },
            dataType:'json',
            success:function(resp){
                if(resp.success){
                    $('#compareModalBody').html(resp.html);
                    var modalEl=new bootstrap.Modal(document.getElementById('compareModal'),{
                        backdrop:'static',keyboard:true
                    });
                    modalEl.show();
                } else {
                    showErrorModal('Compare error: '+resp.message);
                }
            },
            error:function(){
                showErrorModal('Compare error request.');
            }
        });
    });

    // version label context menu
    $(document).on('contextmenu','.open-version', function(e){
        e.preventDefault();
        var versionId=$(this).closest('.version-item').data('version-id');
        $('#versionContextMenu')
          .css({left:e.pageX+'px', top:e.pageY+'px', display:'block'})
          .data('versionId',versionId);
    });
    $(document).on('click',function(){
        $('#versionContextMenu').hide();
    });
    $('#changeVersionLabel').on('click', function(e){
        e.preventDefault();
        $('#versionContextMenu').hide();
        var versionId=$('#versionContextMenu').data('versionId');
        $('#versionLabelModal input[name="new_version_label"]').val('');
        $('#versionLabelModal').data('versionId',versionId).modal('show');
    });
    $('#saveVersionLabelButton').on('click',function(){
        var versionId=$('#versionLabelModal').data('versionId');
        var newLabel=$('#versionLabelModal input[name="new_version_label"]').val().trim();
        $('#versionLabelModal').modal('hide');
        if(!newLabel){
            showErrorModal('Version label cannot be empty.');
            return;
        }
        $.ajax({
            url:'api/update_version_label.php',
            method:'POST',
            data:{
                version_id:versionId,
                new_label:newLabel,
                csrf_token:$('meta[name="csrf-token"]').attr('content')
            },
            dataType:'json',
            success:function(r){
                if(r.success){
                    location.reload();
                } else {
                    showErrorModal('Error updating version label: '+r.message);
                }
            },
            error:function(){
                showErrorModal('Error updating version label request.');
            }
        });
    });

    function showErrorModal(msg){
        $('#errorModalBody').text(msg);
        $('#errorModal').modal('show');
    }

    // ON PAGE LOAD
    if(typeof initialDoc!=='undefined'){
        if(initialDoc.new){
            openNewDocumentTab(initialDoc.title, initialDoc.content);
        } else {
            openExistingDocumentTab(initialDoc.document_id, initialDoc.version, initialDoc.title);
        }
    } else {
        $('#placeholderMessage').show();
    }
});
