<?php
/*
  api/save_document.php
  Reverted "Always New Version" approach
  Final Version: 2025-02-12

  - If save_as_new=1 => we create a brand-new version, incrementing from the last version
  - If force_overwrite=1 => we do that only if user chooses to forcibly overwrite an older version
  - We have incrementVersion(...) for decimals/alpha version strings
*/

require_once '../includes/functions.php';
requireLogin();
require_once '../config/db.php';

header('Content-Type: application/json');

// Validate request
if($_SERVER['REQUEST_METHOD']!=='POST'){
    echo json_encode(['success'=>false,'message'=>'Invalid request method']);
    exit;
}
if(!isset($_POST['csrf_token'])||!verifyToken($_POST['csrf_token'])){
    echo json_encode(['success'=>false,'message'=>'Invalid CSRF token']);
    exit;
}

$document_id=(int)($_POST['document_id']??0);
$version=$_POST['version']??'';
$content=$_POST['content']??'';
$force_overwrite=isset($_POST['force_overwrite']);
$save_as_new=isset($_POST['save_as_new']);
$version_label=isset($_POST['version_label'])?trim($_POST['version_label']):null;

if($document_id<=0||$version===''){
    echo json_encode(['success'=>false,'message'=>'Missing parameters']);
    exit;
}

// fetch doc
$stmtDoc=$pdo->prepare("SELECT * FROM documents WHERE id=? LIMIT 1");
$stmtDoc->execute([$document_id]);
$doc=$stmtDoc->fetch(PDO::FETCH_ASSOC);
if(!$doc){
    echo json_encode(['success'=>false,'message'=>'Document not found']);
    exit;
}

// fetch version row
$stmtVer=$pdo->prepare("SELECT * FROM versions WHERE document_id=? AND version=? LIMIT 1");
$stmtVer->execute([$document_id, $version]);
$versionRow=$stmtVer->fetch(PDO::FETCH_ASSOC);

// function for decimals/alpha
function incrementVersion($old){
    $v=trim($old);
    if(preg_match('/^\d+$/',$v)){
        return (string)((int)$v+1);
    }
    if(preg_match('/^\d+\.\d+$/',$v)){
        $f=(float)$v; $f+=0.1;
        return (string)round($f,1);
    }
    if(preg_match('/(.*?)(\d+)$/',$v,$m)){
        $prefix=$m[1];
        $num=(int)$m[2];
        $num++;
        return $prefix.$num;
    }
    return $v.'.1';
}

try{
    $pdo->beginTransaction();

    if($force_overwrite){
        // Overwrite => only if version row exists
        if(!$versionRow){
            $pdo->rollBack();
            echo json_encode(['success'=>false,'message'=>'Cannot force overwrite: no such version.']);
            exit;
        }
        $sql="UPDATE versions SET content=:cnt, status='saved' WHERE id=:vid";
        $stmt=$pdo->prepare($sql);
        $stmt->execute([
            'cnt'=>$content,
            'vid'=>$versionRow['id']
        ]);
        $pdo->commit();
        echo json_encode(['success'=>true,'new_version'=>$version]);
        exit;

    } elseif($save_as_new){
        // ALWAYS create new version => increment from last version in the DB
        $stmtMax=$pdo->prepare("SELECT version FROM versions WHERE document_id=? ORDER BY id DESC LIMIT 1");
        $stmtMax->execute([$document_id]);
        $rowMax=$stmtMax->fetch(PDO::FETCH_ASSOC);
        $lastVer=$rowMax ? $rowMax['version'] : '1';

        $newVer=incrementVersion($lastVer);
        $stmtIns=$pdo->prepare("
          INSERT INTO versions(document_id, version, content, status, created_at)
          VALUES (:did, :ver, :cnt, 'saved', NOW())
        ");
        $stmtIns->execute([
            'did'=>$document_id,
            'ver'=>$newVer,
            'cnt'=>$content
        ]);
        // update doc => current_version=$newVer
        $stmtDocUpd=$pdo->prepare("UPDATE documents SET current_version=? WHERE id=? LIMIT 1");
        $stmtDocUpd->execute([$newVer,$document_id]);

        $pdo->commit();
        echo json_encode(['success'=>true,'new_version'=>$newVer]);
        exit;

    } else {
        // If not brand new => typically we do 'normal save' => but you wanted "always new version"
        // We'll say if user didn't pass save_as_new => error or handle?
        // For safety, let's just do the same as save_as_new logic:
        // (But you can also do "Document version mismatch" if you want.)
        $stmtMax=$pdo->prepare("SELECT version FROM versions WHERE document_id=? ORDER BY id DESC LIMIT 1");
        $stmtMax->execute([$document_id]);
        $rowMax=$stmtMax->fetch(PDO::FETCH_ASSOC);
        $lastVer=$rowMax ? $rowMax['version']:'1';
        $newVer=incrementVersion($lastVer);

        $stmtIns=$pdo->prepare("
          INSERT INTO versions(document_id, version, content, status, created_at)
          VALUES (:did, :ver, :cnt, 'saved', NOW())
        ");
        $stmtIns->execute([
            'did'=>$document_id,
            'ver'=>$newVer,
            'cnt'=>$content
        ]);
        $stmtDocUpd=$pdo->prepare("UPDATE documents SET current_version=? WHERE id=? LIMIT 1");
        $stmtDocUpd->execute([$newVer,$document_id]);

        $pdo->commit();
        echo json_encode(['success'=>true,'new_version'=>$newVer]);
        exit;
    }

}catch(Exception $ex){
    $pdo->rollBack();
    echo json_encode(['success'=>false,'message'=>'Error saving doc: '.$ex->getMessage()]);
    exit;
}
