<?php
/*
  api/compare_versions.php
  Final Version: 2025-02-10.2

  - Returns a naive line-by-line diff
  - If lines differ => highlight
  - Center gutter => lock scroll + highlight toggle
*/

require_once '../includes/functions.php';
requireLogin();
require_once '../config/db.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success'=>false,'message'=>'Invalid request method.']);
    exit;
}
if (!isset($_POST['csrf_token']) || !verifyToken($_POST['csrf_token'])) {
    echo json_encode(['success'=>false,'message'=>'Invalid CSRF token']);
    exit;
}

$docId1 = isset($_POST['doc_id1']) ? (int)$_POST['doc_id1'] : 0;
$ver1   = isset($_POST['version1']) ? trim($_POST['version1']) : '';
$docId2 = isset($_POST['doc_id2']) ? (int)$_POST['doc_id2'] : 0;
$ver2   = isset($_POST['version2']) ? trim($_POST['version2']) : '';

if ($docId1<=0 || $docId2<=0 || $ver1==='' || $ver2==='') {
    echo json_encode(['success'=>false,'message'=>'Missing doc or version.']);
    exit;
}

// fetch content for doc1/ver1
$stmt = $pdo->prepare("SELECT content FROM versions WHERE document_id=? AND version=? LIMIT 1");
$stmt->execute([$docId1, $ver1]);
$contentA = $stmt->fetchColumn();
// doc2/ver2
$stmt->execute([$docId2, $ver2]);
$contentB = $stmt->fetchColumn();

if ($contentA===false || $contentB===false) {
    echo json_encode(['success'=>false,'message'=>'One or both versions not found.']);
    exit;
}

$linesA = explode("\n", $contentA);
$linesB = explode("\n", $contentB);
$maxLines = max(count($linesA), count($linesB));

ob_start();
?>
<div id="compareDiffContainer" style="width:100%; height:100%; display:flex; overflow:hidden;">
  <div id="compareLeftPanel" style="width:50%; border-right:1px solid #ccc; overflow:auto;">
    <?php
    for($i=0;$i<$maxLines;$i++){
      $la = isset($linesA[$i]) ? rtrim($linesA[$i],"\r") : '';
      $lb = isset($linesB[$i]) ? rtrim($linesB[$i],"\r") : '';
      $class = ($la!==$lb) ? 'diff-line highlight' : 'diff-line';
      ?>
      <div class="<?php echo $class; ?>" style="display:flex;">
        <div class="line-num" style="width:40px; text-align:right; color:#999; margin-right:10px;">
          <?php echo ($i+1); ?>
        </div>
        <div class="line-content"><?php echo htmlspecialchars($la); ?></div>
      </div>
      <?php
    }
    ?>
  </div>
  <div id="compareGutter" style="width:50px; border-right:1px solid #ccc; display:flex; flex-direction:column; align-items:center; justify-content:flex-start; padding:10px;">
    <div class="gutter-icon" id="lockScrollIcon" style="width:30px; height:30px; border:1px solid #bbb; cursor:pointer; margin-bottom:10px; display:flex; align-items:center; justify-content:center;">
      🔒
    </div>
    <div class="gutter-icon" id="highlightIcon" style="width:30px; height:30px; border:1px solid #bbb; cursor:pointer; background:#e0e0e0; display:flex; align-items:center; justify-content:center;">
      🖍️
    </div>
  </div>
  <div id="compareRightPanel" style="width:50%; overflow:auto;">
    <?php
    for($i=0;$i<$maxLines;$i++){
      $la = isset($linesA[$i]) ? rtrim($linesA[$i],"\r") : '';
      $lb = isset($linesB[$i]) ? rtrim($linesB[$i],"\r") : '';
      $class = ($la!==$lb) ? 'diff-line highlight' : 'diff-line';
      ?>
      <div class="<?php echo $class; ?>" style="display:flex;">
        <div class="line-num" style="width:40px; text-align:right; color:#999; margin-right:10px;">
          <?php echo ($i+1); ?>
        </div>
        <div class="line-content"><?php echo htmlspecialchars($lb); ?></div>
      </div>
      <?php
    }
    ?>
  </div>
</div>

<script>
(function(){
  var leftPanel = document.getElementById('compareLeftPanel');
  var rightPanel= document.getElementById('compareRightPanel');
  var lockIcon  = document.getElementById('lockScrollIcon');
  var highlightIcon = document.getElementById('highlightIcon');

  var scrollLocked = false;
  var highlightOn = true;

  lockIcon.addEventListener('click', function(){
    scrollLocked = !scrollLocked;
    lockIcon.style.background = scrollLocked ? '#e0e0e0' : '';
  });
  leftPanel.addEventListener('scroll', function(){
    if(scrollLocked){
      rightPanel.scrollTop = leftPanel.scrollTop;
    }
  });
  rightPanel.addEventListener('scroll', function(){
    if(scrollLocked){
      leftPanel.scrollTop = rightPanel.scrollTop;
    }
  });

  highlightIcon.addEventListener('click', function(){
    highlightOn = !highlightOn;
    highlightIcon.style.background = highlightOn ? '#e0e0e0' : '';
    var leftLines = leftPanel.querySelectorAll('.diff-line');
    var rightLines= rightPanel.querySelectorAll('.diff-line');
    if(highlightOn){
      // re-add highlight if lines differ
      for(var i=0;i<leftLines.length;i++){
        if(leftLines[i].textContent!==rightLines[i].textContent){
          leftLines[i].classList.add('highlight');
          rightLines[i].classList.add('highlight');
        }
      }
    } else {
      // remove highlight
      for(var i=0;i<leftLines.length;i++){
        leftLines[i].classList.remove('highlight');
      }
      for(var i=0;i<rightLines.length;i++){
        rightLines[i].classList.remove('highlight');
      }
    }
  });
})();
</script>
<?php
$html = ob_get_clean();
echo json_encode(['success'=>true,'html'=>$html]);
exit;
